<?php


namespace F\utils;


if(!function_exists('utf8_wordwrap')) {
    function utf8_wordwrap($str, $width, $break)
    {
        if (empty($str) || mb_strlen($str, 'UTF-8') <= $width) return $str;

        $br_width = mb_strlen($break, 'UTF-8');
        $str_width = mb_strlen($str, 'UTF-8');
        $return = '';
        $last_space = false;

        for ($i = 0, $count = 0; $i < $str_width; $i++, $count++) {
            // If we're at a break
            if (mb_substr($str, $i, $br_width, 'UTF-8') == $break) {
                $count = 0;
                $return .= mb_substr($str, $i, $br_width, 'UTF-8');
                $i += $br_width - 1;
                continue;
            }

            // Keep a track of the most recent possible break point
            if (mb_substr($str, $i, 1, 'UTF-8') == " ") {
                $last_space = $i;
            }

            // It's time to wrap
            if ($count >= $width) {
                // There are no spaces to break on!  Going to truncate :(
                if (!$last_space) {
                    $return .= $break;
                    $count = 0;
                } else {
                    // Work out how far back the last space was
                    $drop = $i - $last_space;

                    // Cutting zero chars results in an empty string, so don't do that
                    if ($drop > 0) {
                        $return = mb_substr($return, 0, -$drop);
                    }

                    // Add a break
                    $return .= $break;

                    // Update pointers
                    $i = $last_space + ($br_width - 1);
                    $last_space = false;
                    $count = 0;
                }
            }

            // Add character from the input string to the output
            $return .= mb_substr($str, $i, 1, 'UTF-8');
        }
        return $return;
    }
}

class Str {
    /**
     * @param string $script
     * @return string
     */
    public static function fastCompressJavascript(string $script):string {
        return str_replace(["\n", "\r", '  '], ' ', $script);
    }

    /**
     * @param string $prefix
     * @param array $classes
     * * @return string
     */
    public static function renderBEMClasses(string $prefix, array $classes = []):string {
        $prefix = ' '.$prefix;
        return !empty($classes) ? $prefix.implode($prefix, $classes) : '';
    }

    /**
     * stolen from https://stackoverflow.com/questions/8586141/implode-an-array-with-and-add-and-before-the-last-item
     * @param string[] $items
     * @param ?string $join
     * @return string
     */
    public static function naturalJoin(array $items, ?string $join = null) {
        if(empty($join)) $join = function_exists('__') ? __(' et ', 'fcinq-starter') : ' et ';
        $last  = array_slice($items, -1);
        $first = join(', ', array_slice($items, 0, -1));
        $both  = array_filter(array_merge([$first], $last), 'strlen');
        return join($join, $both);
    }

    /**
     * @param string $str
     * @param string $elmt
     * @param array $classes
     *
     * @return mixed|string
     */
    public static function handleBracketInString($str = '', $elmt = 'i', $classes = []) {
        if(substr_count($str, '[') === substr_count($str, ']')) {
            $classes = !empty($classes) ? ' class="'.implode(' ', $classes).'"' : '';
            return str_replace(['[', ']'], ['<'.$elmt.$classes.'>', '</'.$elmt.'>'], $str);
        }

        return $str;
    }

    /**
     * Renvoie une chaine racourcie
     * @param string $sText: chaine à racourcir
     * @param int $iMaxLength : longueur du racourci
     * @param string $sMessage : message qu'on met à la fin du racourci, exemple 'lire la suite', '[...]' etc...
     * @return string la chaine d'origine si celle ci est moin longue que la taille donnée, sinon la chaine coupée
     */
    public static function wordCut(string $sText, int $iMaxLength, string $sMessage):string {
        $sText = trim(strip_tags($sText));
        $enc   = 'UTF-8';

        // Si le message dépasse la limite, on renvoie juste le message
        $limit = $iMaxLength - mb_strlen($sMessage, $enc);
        if ($limit <= 0) {
            return $sMessage;
        }

        // 2) Si c'est déjà court, on ne touche pas
        if (mb_strlen($sText, $enc) <= $limit) {
            return $sText;
        }

        // 3) On découpe *au plus* à la limite, en UTF-8
        $cut = mb_substr($sText, 0, $limit, $enc);

        // 4) On tente de reculer jusqu’au dernier séparateur "naturel"
        //    (espace, saut de ligne, tab, ponctuation légère)
        $lastBreak = max(
            (int)mb_strrpos($cut, ' ', 0, $enc) ?: -1,
            (int)mb_strrpos($cut, "\n", 0, $enc) ?: -1,
            (int)mb_strrpos($cut, "\t", 0, $enc) ?: -1
        );
        if ($lastBreak > 0) {
            $cut = mb_substr($cut, 0, $lastBreak, $enc);
        }

        // 5) Nettoyage de fin *en une fois* (Unicode-safe)
        //    Retire espaces et une poignée de ponctuations finales si présentes
        $cut = preg_replace('/[ \t\r\n\|\,\."\/\+\:•“”]+$/u', '', $cut);
        $cut = trim($cut);

        return $cut . $sMessage;
    }

    /**
     * @param string $url
     *
     * @return string|null
     */
    public static function extractHostFromURL(string $url):?string {
        $data = parse_url(trim($url));

        return !empty($data['host']) ? $data['host'] : $url;
    }

    /**
     * @param string $search
     * @param string $replace
     * @param string $subject
     * @return string
     */
    public static function replaceFirst(string $search, string $replace, string $subject):string {
        $search = '/'.preg_quote($search, '/').'/';

        return preg_replace($search, $replace, $subject, 1);
    }

    /**
     * @param string $search
     * @param string $replace
     * @param string $subject
     * @return string
     */
    public static function replaceLast(string $search, string $replace, string $subject):string {
        $pos = strrpos($subject, $search);

        if($pos !== false) {
            $subject = substr_replace($subject, $replace, $pos, strlen($search));
        }

        return $subject;
    }

    /**
     * @param string $str
     * @return string
     */
    public static function slugify(string $str):string {
        return sanitize_title($str);
    }

    /**
     * @param string $phone
     *
     * @return bool
     */
    public static function checkPhone(string $phone):bool {
        $phone = preg_replace('/[\s\-\(\)]/', '', $phone);
        return preg_match('/^\+?[0-9]{6,15}$/', $phone);
    }

    /**
     * @param string $url
     *
     * @return string
     */
    public static function cleanUrlHTTP(string $url):string {
        if(strpos($url, 'http://') !== false || strpos($url, 'https://') !== false) {
            return $url;
        } else {
            return 'https://'.$url;
        }
    }

    /**
     * @param string $email
     *
     * @return bool
     */
    public static function checkEmail(string $email):bool {
        if(preg_match("/^([a-zA-Z0-9])+([a-zA-Z0-9+\._-])*@([a-zA-Z0-9_-])+([a-zA-Z0-9\._-]+)+$/", $email)) {
            return true;
        }
        return false;
    }

    /**
     * @param $email string
     *
     * @return string
     */
    public static function getObfuscatedEmailAddress(string $email):string {
        $alwaysEncode = ['.', ':', '@'];
        $result = '';
        // Encode string using oct and hex character codes
        for ($i = 0; $i < strlen($email); $i++) {
            // Encode 25% of characters including several that always should be encoded
            if (in_array($email[$i], $alwaysEncode) || mt_rand(1, 100) < 25) {
                if (mt_rand(0, 1)) {
                    $result .= '&#' . ord($email[$i]) . ';';
                } else {
                    $result .= '&#x' . dechex(ord($email[$i])) . ';';
                }
            } else {
                $result .= $email[$i];
            }
        }

        return $result;
    }


    /**
     * @param $email string
     * @param $label string
     * @param array|null $params
     *
     * @return string
     */
    public static function getObfuscatedEmailLink(string $email, string $label = '', ?array $params = []):string {
        if (!is_array($params)) {
            $params = [];
        }
        // Tell search engines to ignore obfuscated uri
        if (!isset($params['rel'])) {
            $params['rel'] = 'nofollow';
        }
        $neverEncode = ['.', '@', '+']; // Don't encode those as not fully supported by IE & Chrome
        $urlEncodedEmail = '';
        for ($i = 0; $i < strlen($email); $i++) {
            // Encode 25% of characters
            if (!in_array($email[$i], $neverEncode) && mt_rand(1, 100) < 25) {
                $charCode = ord($email[$i]);
                $urlEncodedEmail .= '%';
                $urlEncodedEmail .= dechex(($charCode >> 4) & 0xF);
                $urlEncodedEmail .= dechex($charCode & 0xF);
            } else {
                $urlEncodedEmail .= $email[$i];
            }
        }
        if(empty($label)) $label = self::getObfuscatedEmailAddress($email);
        $obfuscatedEmailUrl = self::getObfuscatedEmailAddress('mailto:' . $urlEncodedEmail);
        $link = '<a href="' . $obfuscatedEmailUrl . '"';
        foreach ($params as $param => $value) {
            $link .= ' ' . $param . '="' . htmlspecialchars($value). '"';
        }
        $link .= '>' . $label . '</a>';
        return $link;
    }


    /**
     * @param array $classes
     * @return string
     */
    public static function renderRSCSSClasses(array $classes = []):string {
        if(!is_array($classes) && !empty($classes)) $classes = [$classes];
        $prefix = ' -';
        $classes = array_filter($classes);
        return !empty($classes) ? $prefix.implode($prefix, array_unique($classes)) : '';
    }


    /**
     * @param string $desktop
     * @param string $mobile
     * @param bool|null $conditionLazyLoad
     */
    public static function renderLazyBKG(string $desktop, string $mobile = '', ?bool $conditionLazyLoad = null):void {
        if(is_null($conditionLazyLoad)) $conditionLazyLoad = !is_admin();
        if(!$conditionLazyLoad) {
            echo ' style="background-image:url('.$desktop.')" ';
        } else {
            echo ' data-bkg-desktop="'.$desktop.'" ';
            if(!empty($mobile)) {
                echo ' data-bkg-mobile="'.$mobile.'" ';
            }
        }
    }

    public static function innerHTML(\DOMNode $element):?string {
        $innerHTML = "";
        $children  = $element->childNodes;

        foreach ($children as $child)  {
            $innerHTML .= $element->ownerDocument->saveHTML($child);
        }

        return $innerHTML;
    }

    public static function removeHTMLComment(string $string):string {
        return preg_replace('/<!--(.|\s)*?-->/', '', $string);
    }

    public static function strContainsArray(string $haystack, array $needles): bool {
        foreach ($needles as $needle) {
            if (str_contains($haystack, $needle)) {
                return true;
            }
        }
        return false;
    }

    public static function getInitials($full_name): string
    {
        $fullName = trim(preg_replace('/\s+/', ' ', $full_name));
        $parts = explode(' ', $fullName);
        $initials = [];
        foreach ($parts as $p) {
            $initials[] = mb_strtoupper(mb_substr($p, 0, 1));
        }

        return implode('.', $initials) . '.';
    }
}
